
 
; TRIAC based motor controller automatic calibration for frequency (50 or 60Hz) requires 220VAC supply
; Soft start either on or off with S1 at GP3.
; With soft start selected, when the speed pot is beyond zero at power up and also when pot brought to 0, soft start ends when motor speed matches pot position
; Current feedback included with a gain setting
; compensation included for phase lag in zero voltage crossing detector

	ERRORLEVEL -302
	ERRORLEVEL -306

	    list      p=12F617            ; list directive to define processor
     #include <p12F617.inc>        ; processor specific variable definitions


     __CONFIG   _CP_OFF & _BOR_OFF & _MCLRE_OFF & _WDT_OFF & _PWRTE_ON & _INTRC_OSC_NOCLKOUT & _IOSCFS_4MHZ & _WRT_OFF

; RAM 
STORE1			equ	H'20'	; delay counter	;
STORE2			equ	H'21'	; delay counter
TEMP			equ	H'22'	; temporary working
SPEED0		equ	H'23'	; speed setting by VR1 ms byte
SPEED1		equ	H'24'	; speed setting by VR1 ls byte
FEEDBACK		equ	H'25'	; current feedback value
GAIN			equ	H'26'	; feedback gain
FEED_SPED0	equ	H'27'	; feedback speed ms
FEED_SPED1	equ	H'28'	; feedback speed ls
SPEEDPOT		equ	H'29'	; speedpot read flag
GATE_OFF		equ	H'2A'	; gate off flag
GATE_ON0		equ	H'2B'	; gate switch on point
GATE_ON1		equ	H'2C'	; gate switch on point
PULSE			equ	H'2D'	; gate pulses flag
SOFTSTART		equ	H'2E'	; soft start flag
SOFT_RUN		equ	H'2F'	; soft start run up value
MEASURE		equ	H'30'	; measure flag for A/D
INTER_STRT		equ	H'31'	; interrupt start counter
GATE			equ	H'32'	; first gate drive signal flag
CYCLE_90		equ	H'33'	; count waveform cycle for 90 degrees
NINETY			equ	H'34'	; 90 degree position
MULT0			equ	H'35'	; multiplier ms byts
MULT1			equ	H'36'	; multiplier ls byte
SOFTSTART_OO	equ	H'37'	; softstart on or off
SS_FLAG		equ	H'38'	; soft start flag
AVERAGE0		equ	H'39'	; ms byte average tally
AVERAGE1		equ	H'3A'	; ls byte
COUNT_AV		equ	H'3B'	; average counter
SOFT_COUNT	equ	H'3C'	; soft start counter

; math routines
TEMPB1		equ H'52'
TEMPB0      		equ H'53'
TEMPD       		equ H'54'   	 ; temporary storage
AARGB3		equ	H'55'
AARGB2      		equ H'56'
AARGB1      		equ H'57'
AARGB0      		equ H'58'	; most significant byte of argument A
BARGB1      		equ H'59'
BARGB0      		equ H'5A'	; most significant byte of argument B
REMB1			equ H'5B'
REMB0       		equ H'5C'   	; most significant byte of remainder
LOOPCOUNT	equ H'5D'   	; loop counter
STO1	    		equ	H'5E'	; AARGB1 storage
STO2			equ	H'5F'	; AARGB2 storage

; All banks RAM
W_TMP			equ	H'70'	; temporary store for w in interrupt
STATUS_TMP	equ	H'71'	; temporary store of status in interrupt 

; ******************************************************************

; start at memory 0

	org		0				; reset vector
	goto	MAIN
; interrupt vector
	org		4
INTERRUPT
	movwf	W_TMP			; w to w_tmp storage
	swapf	STATUS,w		; status to w
	movwf	STATUS_TMP	; status in status_tmp  
	bcf		STATUS,RP0	; select memory bank 0

	btfsc	CYCLE_90,1		; count waveform cycle for 90degrees, if clear get timer0 count from edge to edge
	goto	RUN

;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
CALIBRATION; at power up 

; calibrate values for multiplier and ninety degrees value
	btfss	INTCON,GPIF		; GP2 port change 
	goto	RECLAIM
	movf	GPIO,w				; read to clear mismatch plus check GP2 Level
	bcf		INTCON,GPIF
	btfsc 	GPIO,2				; only calibrate with a going low GP2 level (ie a full cycle of mains waveform)
	goto 	RECLAIM

; If CYCLE_90 ,0 is clear then clear timer1
	btfss	CYCLE_90,0		; count waveform cycle for 90 degrees
	goto	WAIT_NEXT_EDGE
; get timer1 count
	bcf		T1CON,0		; stop timer 1

;  timer1 count from edge to edge divide by 16
	bcf		STATUS,C		; clear carry
	rrf		TMR1H,f			; 
	rrf		TMR1L,f			; /2
; store in multiplier
	movf	TMR1H,w		; most significant store in MULT0
	movwf	MULT0
	movf	TMR1L,w		; least significant
	movwf	MULT1
	
; take away D64 due to phase lag in zero voltage crossing (400us) and 100us gate pulse (8us x 64 = 512us)
	movlw	D'64'
	subwf	MULT1,f
	btfss	STATUS,C		; if carry then decrease ms byte
	decf	MULT0,f
; get PR2 value by dividing TIMER1  by a further 8
	bcf		STATUS,C		; clear carry
	rrf		TMR1H,f			; 
	rrf		TMR1L,f			; /2
	bcf		STATUS,C		; clear carry
	rrf		TMR1H,f			; 
	rrf		TMR1L,f			; /4
	bcf		STATUS,C		; clear carry
	rrf		TMR1H,f			; 
	rrf		TMR1L,f			; /8
	bcf		STATUS,C		; clear carry
	rrf		TMR1H,f			; 
	rrf		TMR1L,w		; /16
	movwf	NINETY			; place 90 degree value
	bsf		T1CON,0		; start timer1 again
; enable interrupts
	bsf		STATUS,RP0	; select memory bank 1
	bsf		PIE1,TMR1IE		; timer 1 overflow interrupt
	bcf		STATUS,RP0	; select memory bank 0
	bsf		INTCON,PEIE	; enable periperal interrupts 
	bcf		PIR1,TMR1IF		; timer 1 interrupt flag
	bsf		CYCLE_90,1		; enable normal run
	goto	RECLAIM
WAIT_NEXT_EDGE
	clrf 		TMR1L			; cleared on edge
	clrf		TMR1H
	bsf		CYCLE_90,0		; set bit 0
	goto	RECLAIM
;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

; check which interrupt	
RUN	
	btfsc	PIR1,TMR1IF		; timer 1 interrupt flag
	goto	T1	
	btfss	INTCON,GPIF	; GP2 port change interrupt flag 
	goto	RECLAIM

EDGE
;  (GP2)	
	movf	GPIO,w			; clear mismatch
; check level
	btfss	GPIO,2
	bsf		MEASURE,0		; set flag only on one zero crossing level
	bcf		INTCON,GPIF	; flag cleared

; clear timer0
	clrf		TMR0			; set at 0	
	clrw
	movwf	GPIO			; gate off at bit 5 (GP5)
	clrf		PULSE
	clrf		GATE
; preload timer with 65536 minus the speed setting with feedback

	clrf		TMR1L 			; Clear Low byte, Ensures no rollover into TMR1H
	movf	GATE_ON0,w	; Value to load into TMR1H
	movwf	TMR1H			 ; Write High byte
	movf	GATE_ON1,w	 ; Value to load into TMR1L
	movwf	TMR1L	 		;  Write Low byte
	bcf		PIR1,TMR1IF		; timer 1 interrupt flag cleared
	clrf		PULSE			; flag for gate pulses
	goto	RECLAIM

T1 ; timer1 overflow	
	bcf		PIR1,TMR1IF		; timer 1 interrupt flag

	btfss	SPEEDPOT,0	; speed pot read flag. If  flag is set as having been read then continue, otherwise exit
	goto	RECLAIM

; check if first flag clear
	btfss	GATE,0			; trigger on first gate 

	goto	WIDER	

CHK_PULSES
	btfsc	PULSE,0
	goto	PULSE_OFF

; drive gate at timer overflow
; multiple gate pulses for trigger at between 0-90 degrees
; single gate pulse for when trigger is between 90 and 180 degrees
; timer0 has 64us per count
; Lag due to filter on zero crossing is 8 counts 
; value in ninety compensates for zero voltage crossing lag of around 500us. Timer0 counts at 64us about 8 counts
	movf	TMR0,w
	subwf	NINETY,w		; D'70'; for 50Hz	D'57'; for 60Hz
	btfss	STATUS,C		; bypass gate on if timer greater
	goto	INC_PULSE	
; check if startup period has ended
	movf	INTER_STRT	,w	; interrupt start counter
	btfsc	STATUS,Z		; if zero then can drive gate
	goto	DRIVE
	decfsz	INTER_STRT,f	; decrease until zero
	goto	WIDER			;  changed from INC_PULSE		; bypass drive
DRIVE
	movlw	B'00000000'		; gate off if flag clear
	btfsc	GATE_OFF,0	; gate off flag. If set allow gate drive
	movlw	B'00100000'		; set bit 5
	movwf	GPIO			; gate drive 
INC_PULSE
	incf		PULSE,f			; once first fired alternate on and off
	bsf		GATE,0			; set after first gate pulse

; reload for 40us
	clrf		TMR1L 			; Clear Low byte, Ensures no rollover into TMR1H
	movlw	H'FF'
	movwf	TMR1H
	movlw	H'FE'			; H' FE'  for 40us (use H'DF' for 300us wider single pulse test)**
	movwf	TMR1L
	bcf		PIR1,TMR1IF		; timer 1 interrupt flag cleared
	goto	RECLAIM

WIDER
	movlw	B'00000000'		; gate off if flag clear
	btfsc	GATE_OFF,0	; gate off flag. If set allow gate drive
	movlw	B'00100000'		; set bit 5
	movwf	GPIO			; gate drive
	incf		PULSE,f			; once first fired alternate on and off
	bsf		GATE,0			; set after first gate pulse

; reload for 100us gate on
	clrf		TMR1L 			; Clear Low byte, Ensures no rollover into TMR1H
	movlw	H'FF'
	movwf	TMR1H
	movlw	H'F7'
	movwf	TMR1L
	bcf		PIR1,TMR1IF		; timer 1 interrupt flag cleared
	goto	RECLAIM

PULSE_OFF
	clrf		PULSE			; remark out to prevent multiple pulses. ie a single 40us pulse < 90 degrees, 100us pulse >90 degrees. Change 40us pulse see above)**
	clrw
	movwf	GPIO			; gate off
; reload for 200us gate off
	clrf		TMR1L 			; Clear Low byte, Ensures no rollover into TMR1H
	movlw	H'FF'
	movwf	TMR1H
	movlw	H'EB'
	movwf	TMR1L
	bcf		PIR1,TMR1IF		; timer 1 interrupt flag cleared
	
; end of interrupt reclaim w and status 

RECLAIM
	swapf	STATUS_TMP,w	; status temp storage to w
	movwf	STATUS			; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf  	W_TMP,w		; swap bits and into w register
	retfie					; return from interrupt

; ***********************************************************
		
MAIN
; set oscillator calibration
	bsf		STATUS,RP0	; bank 1
        movlw   0x00      			 ; set oscillator to factory calibrated frequency 
        movwf   OSCTUNE
	bcf		STATUS,RP0
; set inputs/outputs
	movlw	B'00000000'
	movwf	GPIO			; ports low
	movlw	B'00000111'		; comparators off
	movwf	CMCON0
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'00000000'		; pullups off
	movwf	WPU
	movlw	B'00011111'		; outputs/inputs set 
	movwf	TRISIO			; port data direction register
	movlw	B'00000101'		; settings (pullups disabled) timer0 /64 prescaler (bit 6: 1 is rising edge, 0 is falling edge interrupt) 4V and 1V triggers
	movwf	OPTION_REG
; analog inputs, A/D
	movlw	B'01011011'
	movwf	ANSEL			; digital I/O and analog 

	bcf		STATUS,RP0	; select memory bank 0
	movlw	B'00000000'		; channel 0, left justified, VDD ref etc

;bits 3-2 CHS1:CHS0: Analog Channel Select bits
;00 = Channel 00 (AN0) FEEDBACK Gain
;01 = Channel 01 (AN1) SPEED POT
;11 = Channel 03 (AN3) CURRENT

	movwf	ADCON0
	bsf		ADCON0,0	; A/D on

;Timer1 on
	movlw 	B'00110001'		; divide by 8 for 8us per count
	movwf	T1CON

;  initial states
	clrf		SPEED0		; speed setting by VR1 ms byte
	clrf		SPEED1		; speed setting by VR1 ls byte
	clrf		SOFTSTART		; soft start flag cleared at startup and after speed pot returned to minimum
	movlw	H'FF'
	movwf	SOFT_RUN		; start at end of cycle ramping up to  SPEED1
	clrf		SPEEDPOT		; speed pot read flag
	clrf		TMR1L			; timer1 clear
	clrf		TMR1H
	movlw	D'10'
	movwf	INTER_STRT		; interrupt start counter
	clrf		CYCLE_90		; count waveform cycle for 90 degrees
	clrf		GATE_OFF		; gate off flag
	clrf		MEASURE		; measure A/D flag
; clear averaging
	clrf		AVERAGE0		; ms byte average tally
	clrf		AVERAGE1		; ls byte
	clrf		COUNT_AV		; average counter	

; Startup timer
	movlw	D'10'
	movwf	TEMP			;  cycles of delay period
STRT_DEL
; approx 100ms
	movlw	D'115'			; delay value (D115 gives 100ms) 
	call		DEL_MORE
	decfsz	TEMP,f
	goto	STRT_DEL		; startup delay 

; Interrupts
	bsf		STATUS,RP0	; select memory bank 1
	bsf		IOC,2			; port change on GP2		
	bcf		STATUS,RP0	; select memory bank 0

	bsf		INTCON,GPIE	; enable port change interrrupt
	bcf		INTCON,GPIF	;  flag	 
	bsf 		INTCON,GIE		; set global interrupt enable 

UPDATE

; wait for measure flag
	btfss	MEASURE,0		;
	goto	UPDATE
	clrf		MEASURE		; cleared once detected from interrupt

; check softstart
	movlw	D'1'				; softstart off
	btfss	GPIO,3			; if low set softstart flag otherwise clear
	clrw					; softstart on
	movwf	SOFTSTART_OO; on or off
	
	movf	SOFTSTART_OO,w	; if zero then softstart
	movlw	D'1'
	btfss	STATUS,Z
	movwf	SOFTSTART		; set at 1 so feedback control operates when softstart is not selected

; read VR1, VR2 and feedback
; bits 3-2 CHS1:CHS0: Analog Channel Select bits

; Read feedback gain trim pot
; FEEDBACK Gain
;00 = Channel 0 (AN0) 
	bcf		ADCON0,2
	bcf		ADCON0,3		; channel 0
	call		DELAYX
	call		ACQUIRE_AD

; averaging
; if COUNT_AV=0, average values and place in GAIN
	movf	COUNT_AV,w
	btfsc	STATUS,Z
	goto 	AV_END
; do averaging
	movf	ADRESH,w
	addwf	AVERAGE1,f
	btfsc	STATUS,C		; if over then increase ms byte
	incf		AVERAGE0,f
	decfsz	COUNT_AV,f
	goto	FEEDB

AV_END
; divide by 16 for average value
	bcf		STATUS,C
	rrf		AVERAGE0,f
	rrf		AVERAGE1,f		; divide by 2
	bcf		STATUS,C
	rrf		AVERAGE0,f
	rrf		AVERAGE1,f 	; by 4
	bcf		STATUS,C
	rrf		AVERAGE0,f
	rrf		AVERAGE1,f		; by 8
	bcf		STATUS,C
	rrf		AVERAGE0,f
	rrf		AVERAGE1,f		; by 16
	bcf		STATUS,C
	rrf		AVERAGE1,w	; 0-127 for gain value
	movwf	GAIN
; restart average
	movlw	D'16'
	movwf	COUNT_AV
	clrf		AVERAGE0
	clrf		AVERAGE1

FEEDB
;11 = Channel 03 (AN3) 
;FEEDBACK CURRENT
	bsf		ADCON0,2		; 
	bsf		ADCON0,3		; channel 3
	call		DELAYX
	call		ACQUIRE_AD
	movf	ADRESH,w
	movwf	FEEDBACK		; feedback from current transformer

	movf	SOFTSTART_OO,w
	btfss	STATUS,Z		; when zero run softstart if required
	goto	SPEED_POT_AD

; soft start with zero current
	movf	FEEDBACK,w
	sublw	D'1'				; feedback needs to be 2 or more for non-zero current detect
	btfss	STATUS,C
	goto	SPEED_POT_AD

; decrease SOFT_COUNT. If zero then restart soft start  (ie clear SS_FLAG)

	movf	SOFT_COUNT,w
	btfsc	STATUS,Z
	goto	CSSF
	decfsz	SOFT_COUNT,f
	goto	CSSF
	clrf		SS_FLAG

CSSF
; check soft start flag 
	btfsc	SS_FLAG,0		; if soft start has started, bypass
	goto	SPEED_POT_AD
SOFT_RUN1

	movlw	D'11'
	movwf	SOFT_COUNT	; soft start cycle counter
	clrf		SOFTSTART		; start softstart
	movlw	D'255'			; 135 degrees max (using D191)
	movwf	SOFT_RUN		; start at end of cycle ramping up to  SPEED1

; set flag so doesn't continually reset to begin softstart
	bsf		SS_FLAG,0

SPEED_POT_AD	
; SPEED POT
;01 = Channel 01 (AN1) 
	bsf		ADCON0,2		; channel 1
	bcf		ADCON0,3		;
	call		DELAYX
	call		ACQUIRE_AD

	bsf		GATE_OFF,0	; gate off flag, set so gate can be driven
	movf	ADRESH,w
; if 0 then clear gate off flag
	btfsc	STATUS,Z
	clrf		GATE_OFF		; gate off
	comf	ADRESH,w		; 8-bit msb inverted
	movwf	SPEED1		; ls byte

	movf	SOFTSTART_OO,w
	btfss	STATUS,Z		; when zero run softstart if required
	goto	LS_BITS_RUN

; if softstart is clear, reduce startup from FF to SPEED1  

	movf	SOFTSTART,w
	btfss	STATUS,Z
	goto	LS_BITS_RUN

; compare with SPEED1
	movf	SPEED1,w
	subwf	SOFT_RUN,w
	btfss	STATUS,C
	goto	END_SOFT		; use unchanged SPEED1	

; check if SOFT_RUN is zero
	movf	SOFT_RUN,w
	btfsc	STATUS,Z		; if zero then use unchanged SPEED1 and set SOFTSTART flag to end soft start
	goto	END_SOFT

; decrease by one or more
	movlw	D'4'
	subwf	SOFT_RUN,f	
	btfsc	STATUS,Z		
	goto	END_SOFT		; if zero then use unchanged SPEED1 and set SOFTSTART flag to end soft start
; check if <0
	btfss	STATUS,C	
	goto	END_SOFT		; if zero then use unchanged SPEED1 and set SOFTSTART flag to end soft start

TRANSFER
; transfer SOFT_RUN to SPEED1
	movf	SOFT_RUN,w
	movwf	SPEED1
	goto	LS_BITS_RUN
	
END_SOFT; end of soft start
	movlw	D'01'
	movwf	SOFTSTART
	clrf		SS_FLAG		; soft start flag
LS_BITS_RUN
; get ls bits from A/D
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w		; bits 9 and 10 lsb
	bcf		STATUS,RP0	; select memory bank 0
	movwf	TEMP			; lsb
	comf	TEMP,f			; complement value for reverse direction
	clrf		SPEED0	

; shift TEMP into SPEED and then SPEED0 for 10-bit 1024 value range
	rlf		TEMP,f
	rlf		SPEED1,f
	rlf		SPEED0,f	
	rlf		TEMP,f
	rlf		SPEED1,f
	rlf		SPEED0,f	
;  range in 8us steps

; due to zero voltage phase lag, set maximum value so pulses are stopped 400us plus the 100us gate pulse before next zero crossing detection

	movf	SPEED0,w		; ms byte of speed setting
	movwf	AARGB0	
	movf	SPEED1,w		; ls byte
	movwf	AARGB1

; multiply by initial calibration values in MULT0 and MULT1 then divide by 1024
	movf	MULT0,w
	movwf	BARGB0
	movf	MULT1,w
	movwf	BARGB1

	movwf	BARGB1
	call 		FXM1616U		; multiply
; result in AARGB0,1,2,3
; divide by 1024 by using AARGB1 and AARGB2 for divide by 256 and rotating right by two steps
	bcf		STATUS,C
	rrf		AARGB1,f
	rrf		AARGB2,f
	bcf		STATUS,C
	rrf		AARGB1,w
	movwf	SPEED0		; speed ms byte
	rrf		AARGB2,w		; place in SPEED
	movwf	SPEED1		; ls byte
;....................................................................................................................................................
; do feedback calculations. 

; FEEDBACK is 8-bit value of current
; GAIN is amount of feedback to be applied to speed
	clrf		AARGB0		; ms bytes cleared
	clrf		BARGB0

; clear GAIN when in softstart
	movf	SOFTSTART,w
	btfss	STATUS,Z		; if softstart use 0 for gain
	goto	GAIN_GAIN
	clrw	; use 0 for gain
	goto	GAIN0
GAIN_GAIN; use gain value
; if GAIN < 5 use 0
	movlw	D'5'
	subwf	GAIN,w			; load ls bytes
	btfss	STATUS,C
	clrw
	
GAIN0
	movwf	AARGB1
	movf	FEEDBACK,w
	movwf	BARGB1
	call 		FXM1616U		; multiply
; result in AARGB,2,3
; divide by  rotating right 
	bcf		STATUS,C
	rrf		AARGB2,f
	rrf		AARGB3,f		; /2
	bcf		STATUS,C
	rrf		AARGB2,f		;  ms byte
	rrf		AARGB3,f		;  ls byte /4
	bcf		STATUS,C
	rrf		AARGB2,f
	rrf		AARGB3,f		; /8
	bcf		STATUS,C
	rrf		AARGB2,f		;  ms byte
	rrf		AARGB3,f		;  ls byte /16

; subtract from SPEED0,1

	movf   	AARGB2,w 	 	 ; high byte 
    	subwf   	SPEED0,f    		 ; subtract high bytes
    
    	movf	AARGB3,w 		; low byte
    	subwf   	SPEED1,f     		; subtract low bytes
	btfss   	STATUS,C    	 	; carry check
	decf	SPEED0,f
	btfss   	SPEED0,7     	; bypass if SPEED> AARGB
	goto    	SUB_VAL        	; use subtracted value

; clear SPEED
	clrf		SPEED0
	clrf		SPEED1
SUB_VAL; subtracted value or 0
;......................................................................................................................................................
;  65536 minus speed for gate_on
	movf	SPEED0,w
	sublw	H'FF'			; ms byte subtract
	movwf	TEMP
	movf	SPEED1,w
	sublw	H'FF'
;
; stop interrupt
	bcf		INTCON,GIE
	movwf	GATE_ON1		; ls byte
	btfss	STATUS,C
	decf	TEMP,f

	movf	TEMP,w
	movwf	GATE_ON0		; ms byte
; re-allow interrupt
	bsf		INTCON,GIE

	bsf		SPEEDPOT,0	; speed pot read flag. Set flag as having been read

	goto	UPDATE
; *****************************************************************************************************************************************************
; subroutines

; subroutine to wait for A/D conversion
ACQUIRE_AD
	bsf		ADCON0,1	; GO/DONE bit start conversion
WAIT_CONV
	btfsc	ADCON0,1	; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV
	return

; delay loop 

DELAYms; approx 20ms
	movlw	D'23'		; delay value 
DEL_MORE
	movwf	STORE1		; STORE1 is number of loops value
LOOP8	
	movlw	D'117'
	movwf	STORE2		; STORE2 is internal loop value	
LOOP9
	decfsz	STORE2,f
	goto	LOOP9
	decfsz	STORE1,f
	goto	LOOP8
	return

DELAYX
	movlw	D'30'		; D30
	movwf	STORE1		; STORE1 is  loop value	
LOOP1
	decfsz	STORE1,f
	goto	LOOP1
	return

; multiply
;
;       Input:  fixed point arguments in AARG and BARG
;
;       Output: product AARGxBARG in AARG
;

;       16x16 Bit Unsigned Fixed Point Multiply 

;       Input:  16 bit unsigned fixed point multiplicand in AARGB0
;               16 bit unsigned fixed point multiplier in BARGB0

;       Use:    CALL    FXM1616U

;       Output: 32 bit unsigned fixed point product in AARGB0



FXM1616U 
	        CLRF    AARGB2          ; clear partial product
                CLRF    AARGB3
                MOVF    AARGB0,W
                MOVWF   TEMPB0
                MOVF    AARGB1,W
                MOVWF   TEMPB1
                MOVLW   H'08'
                MOVWF   LOOPCOUNT
LOOPUM1616A     RRF     BARGB1, F
                BTFSC   STATUS,C
                GOTO    ALUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616A
                MOVWF   LOOPCOUNT
LOOPUM1616B     RRF     BARGB0, F
                BTFSC   STATUS,C
                GOTO    BLUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616B
                CLRF    AARGB0
                CLRF    AARGB1
                RETLW   H'00'
BLUM1616NAP     BCF     STATUS,C
                GOTO    BLUM1616NA
ALUM1616NAP     BCF     STATUS,C
                GOTO    ALUM1616NA
ALOOPUM1616     RRF     BARGB1, F
                BTFSS   STATUS,C
                GOTO    ALUM1616NA
                MOVF    TEMPB1,W
                ADDWF   AARGB1, F
                MOVF    TEMPB0,W
                BTFSC   STATUS,C
                INCFSZ  TEMPB0,W
                ADDWF   AARGB0, F
ALUM1616NA      RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                DECFSZ LOOPCOUNT, F
                GOTO   ALOOPUM1616
                MOVLW  H'08'
                MOVWF  LOOPCOUNT
BLOOPUM1616     RRF    BARGB0, F
                BTFSS  STATUS,C
                GOTO   BLUM1616NA
                MOVF   TEMPB1,W
                ADDWF  AARGB1, F
                MOVF   TEMPB0,W
                BTFSC  STATUS,C
                INCFSZ TEMPB0,W
                ADDWF  AARGB0, F
BLUM1616NA      RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                RRF    AARGB3, F
                DECFSZ LOOPCOUNT, F
                GOTO   BLOOPUM1616
                RETURN


	end
